/*!
 * @nuxt/config v2.16.3 (c) 2016-2023
 * Released under the MIT License
 * Repository: https://github.com/nuxt/nuxt.js
 * Website: https://nuxtjs.org
*/
'use strict';

const lodash = require('lodash');
const stdEnv = require('std-env');
const utils = require('@nuxt/utils');
const path = require('path');
const fs = require('fs');
const defu = require('defu');
const consola = require('consola');
const destr = require('destr');
const ufo = require('ufo');
const dotenv = require('dotenv');
const rc = require('rc9');

function _interopNamespaceDefault(e) {
  const n = Object.create(null);
  if (e) {
    for (const k in e) {
      if (k !== 'default') {
        const d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    }
  }
  n.default = e;
  return Object.freeze(n);
}

const rc__namespace = /*#__PURE__*/_interopNamespaceDefault(rc);

const _app = () => ({
  vue: {
    config: {
      silent: void 0,
      // = !dev
      performance: void 0
      // = dev
    }
  },
  vueMeta: null,
  head: {
    meta: [],
    link: [],
    style: [],
    script: []
  },
  fetch: {
    server: true,
    client: true
  },
  plugins: [],
  extendPlugins: null,
  css: [],
  layouts: {},
  ErrorPage: null,
  loading: {
    color: "black",
    failedColor: "red",
    height: "2px",
    throttle: 200,
    duration: 5e3,
    continuous: false,
    rtl: false,
    css: true
  },
  loadingIndicator: "default",
  pageTransition: {
    name: "page",
    mode: "out-in",
    appear: false,
    appearClass: "appear",
    appearActiveClass: "appear-active",
    appearToClass: "appear-to"
  },
  layoutTransition: {
    name: "layout",
    mode: "out-in"
  },
  features: {
    store: true,
    layouts: true,
    meta: true,
    middleware: true,
    transitions: true,
    deprecations: true,
    validate: true,
    asyncData: true,
    fetch: true,
    clientOnline: true,
    clientPrefetch: true,
    componentAliases: true,
    componentClientOnly: true
  }
});

const _common = () => ({
  // Env
  dev: Boolean(stdEnv.isDevelopment),
  test: Boolean(stdEnv.isTest),
  debug: void 0,
  // = dev
  env: {},
  createRequire: void 0,
  // Target
  target: utils.TARGETS.server,
  // Rendering
  ssr: true,
  // Mode (deprecated)
  mode: void 0,
  // Modern
  modern: void 0,
  // Modules
  modules: [],
  buildModules: [],
  _modules: [],
  globalName: void 0,
  globals: {
    id: (globalName) => `__${globalName}`,
    nuxt: (globalName) => `$${globalName}`,
    context: (globalName) => `__${globalName.toUpperCase()}__`,
    pluginPrefix: (globalName) => globalName,
    readyCallback: (globalName) => `on${lodash.capitalize(globalName)}Ready`,
    loadedCallback: (globalName) => `_on${lodash.capitalize(globalName)}Loaded`
  },
  // Server
  serverMiddleware: [],
  // Dirs and extensions
  _nuxtConfigFile: void 0,
  srcDir: void 0,
  buildDir: ".nuxt",
  modulesDir: [
    "node_modules"
  ],
  dir: {
    assets: "assets",
    app: "app",
    layouts: "layouts",
    middleware: "middleware",
    pages: "pages",
    static: "static",
    store: "store"
  },
  extensions: [],
  styleExtensions: ["css", "pcss", "postcss", "styl", "stylus", "scss", "sass", "less"],
  alias: {},
  // Ignores
  ignoreOptions: void 0,
  ignorePrefix: "-",
  ignore: [
    "**/*.test.*",
    "**/*.spec.*"
  ],
  // Watch
  watch: [],
  watchers: {
    rewatchOnRawEvents: void 0,
    webpack: {
      aggregateTimeout: 1e3
    },
    chokidar: {
      ignoreInitial: true
    }
  },
  // Editor
  editor: void 0,
  // Hooks
  hooks: null,
  // runtimeConfig
  privateRuntimeConfig: {},
  publicRuntimeConfig: {}
});

const build = () => ({
  quiet: Boolean(stdEnv.isCI || stdEnv.isTest),
  analyze: false,
  profile: process.argv.includes("--profile"),
  extractCSS: false,
  cssSourceMap: void 0,
  ssr: void 0,
  parallel: false,
  cache: false,
  standalone: false,
  publicPath: "/_nuxt/",
  serverURLPolyfill: "url",
  filenames: {
    // { isDev, isClient, isServer }
    app: ({ isDev, isModern }) => isDev ? `[name]${isModern ? ".modern" : ""}.js` : `[contenthash:7]${isModern ? ".modern" : ""}.js`,
    chunk: ({ isDev, isModern }) => isDev ? `[name]${isModern ? ".modern" : ""}.js` : `[contenthash:7]${isModern ? ".modern" : ""}.js`,
    css: ({ isDev }) => isDev ? "[name].css" : "css/[contenthash:7].css",
    img: ({ isDev }) => isDev ? "[path][name].[ext]" : "img/[name].[contenthash:7].[ext]",
    font: ({ isDev }) => isDev ? "[path][name].[ext]" : "fonts/[name].[contenthash:7].[ext]",
    video: ({ isDev }) => isDev ? "[path][name].[ext]" : "videos/[name].[contenthash:7].[ext]"
  },
  loaders: {
    file: { esModule: false },
    fontUrl: { esModule: false, limit: 1e3 },
    imgUrl: { esModule: false, limit: 1e3 },
    pugPlain: {},
    vue: {
      transformAssetUrls: {
        video: "src",
        source: "src",
        object: "src",
        embed: "src"
      }
    },
    css: {
      esModule: false,
      modules: {
        compileType: "icss"
      }
    },
    cssModules: {
      esModule: false,
      modules: {
        localIdentName: "[local]_[hash:base64:5]"
      }
    },
    less: {},
    sass: {
      sassOptions: {
        indentedSyntax: true
      }
    },
    scss: {},
    stylus: {},
    vueStyle: {}
  },
  styleResources: {},
  plugins: [],
  terser: {},
  hardSource: false,
  aggressiveCodeRemoval: false,
  optimizeCSS: void 0,
  optimization: {
    runtimeChunk: "single",
    minimize: void 0,
    minimizer: void 0,
    splitChunks: {
      chunks: "all",
      automaticNameDelimiter: "/",
      cacheGroups: {}
    }
  },
  splitChunks: {
    layouts: false,
    pages: true,
    commons: true
  },
  corejs: "auto",
  babel: {
    configFile: false,
    babelrc: false,
    cacheDirectory: void 0
  },
  transpile: [],
  // Name of NPM packages to be transpiled
  postcss: {
    // Use the default postcss config: stage 2
    // https://cssdb.org/#staging-process
    postcssOptions: {}
  },
  html: {
    minify: {
      collapseBooleanAttributes: true,
      decodeEntities: true,
      minifyCSS: true,
      minifyJS: true,
      processConditionalComments: true,
      removeEmptyAttributes: true,
      removeRedundantAttributes: true,
      trimCustomFragments: true,
      useShortDoctype: true
    }
  },
  template: void 0,
  templates: [],
  watch: [],
  devMiddleware: {
    // stats will be printed by webapckbar StateReporter
    stats: "none"
  },
  hotMiddleware: {},
  stats: {
    excludeAssets: [
      /.map$/,
      /index\..+\.html$/,
      /vue-ssr-(client|modern)-manifest.json/
    ]
  },
  friendlyErrors: true,
  additionalExtensions: [],
  warningIgnoreFilters: [],
  followSymlinks: false,
  loadingScreen: {},
  indicator: {
    position: "bottom-right",
    backgroundColor: "#2E495E",
    color: "#00C48D"
  }
});

const messages = () => ({
  loading: "Loading...",
  error_404: "This page could not be found",
  server_error: "Server error",
  nuxtjs: "Nuxt",
  back_to_home: "Back to the home page",
  server_error_details: "An error occurred in the application and your page could not be served. If you are the application owner, check your logs for details.",
  client_error: "Error",
  client_error_details: "An error occurred while rendering the page. Check developer tools console for details."
});

const modes = () => ({
  [utils.MODES.universal]: {
    build: {
      ssr: true
    },
    render: {
      ssr: true
    }
  },
  [utils.MODES.spa]: {
    build: {
      ssr: false
    },
    render: {
      ssr: false
    }
  }
});

const render = () => ({
  bundleRenderer: {
    shouldPrefetch: () => false,
    shouldPreload: (fileWithoutQuery, asType) => ["script", "style"].includes(asType),
    runInNewContext: void 0
  },
  crossorigin: void 0,
  resourceHints: true,
  ssr: void 0,
  ssrLog: void 0,
  http2: {
    push: false,
    shouldPush: null,
    pushAssets: null
  },
  static: {
    prefix: true
  },
  compressor: {
    threshold: 0
  },
  etag: {
    weak: false
  },
  csp: false,
  dist: {
    // Don't serve index.html template
    index: false,
    // 1 year in production
    maxAge: "1y"
  },
  // https://github.com/nuxt/serve-placeholder
  fallback: {
    dist: {},
    static: {
      skipUnknown: true,
      handlers: {
        ".htm": false,
        ".html": false
      }
    }
  }
});

const router = () => ({
  mode: "history",
  base: "/",
  routes: [],
  routeNameSplitter: "-",
  middleware: [],
  linkActiveClass: "nuxt-link-active",
  linkExactActiveClass: "nuxt-link-exact-active",
  linkPrefetchedClass: false,
  extendRoutes: null,
  scrollBehavior: null,
  parseQuery: false,
  stringifyQuery: false,
  fallback: false,
  prefetchLinks: true,
  prefetchPayloads: true,
  trailingSlash: void 0
});

const server = ({ env = {} } = {}) => ({
  https: false,
  port: env.NUXT_PORT || env.PORT || env.npm_package_config_nuxt_port || 3e3,
  host: env.NUXT_HOST || env.HOST || env.npm_package_config_nuxt_host || "localhost",
  socket: env.UNIX_SOCKET || env.npm_package_config_unix_socket,
  timing: false
});

const cli = () => ({
  badgeMessages: [],
  bannerColor: "green"
});

const generate = () => ({
  dir: "dist",
  routes: [],
  exclude: [],
  concurrency: 500,
  ignoreEnv: false,
  interval: 0,
  subFolders: true,
  fallback: "200.html",
  crawler: true,
  manifest: true,
  nojekyll: true,
  cache: {
    ignore: [],
    globbyOptions: {
      gitignore: true
    }
  },
  staticAssets: {
    base: void 0,
    // Default: "/_nuxt/static:
    versionBase: void 0,
    // Default: "_nuxt/static/{version}""
    dir: "static",
    version: void 0
    // Default: "{timeStampSec}"
  }
});

const defaultNuxtConfigFile = "nuxt.config";
function getDefaultNuxtConfig(options = {}) {
  if (!options.env) {
    options.env = process.env;
  }
  return {
    ..._app(),
    ..._common(),
    build: build(),
    messages: messages(),
    modes: modes(),
    render: render(),
    router: router(),
    server: server(options),
    cli: cli(),
    generate: generate()
  };
}

function getNuxtConfig(_options) {
  var _a;
  if (_options.__normalized__) {
    return _options;
  }
  const options = Object.assign({}, _options);
  options.__normalized__ = true;
  if (options.loading === true) {
    delete options.loading;
  }
  if (options.router && options.router.middleware && !Array.isArray(options.router.middleware)) {
    options.router.middleware = [options.router.middleware];
  }
  if (options.router && typeof options.router.base === "string") {
    options._routerBaseSpecified = true;
  }
  if (options.router && typeof options.router.scrollBehavior !== "undefined") {
    consola.warn("`router.scrollBehavior` property is deprecated in favor of using `~/app/router.scrollBehavior.js` file, learn more: https://nuxtjs.org/api/configuration-router#scrollbehavior");
  }
  if (typeof options.transition !== "undefined") {
    consola.warn("`transition` property is deprecated in favor of `pageTransition` and will be removed in Nuxt 3");
    options.pageTransition = options.transition;
    delete options.transition;
  }
  if (typeof options.pageTransition === "string") {
    options.pageTransition = { name: options.pageTransition };
  }
  if (typeof options.layoutTransition === "string") {
    options.layoutTransition = { name: options.layoutTransition };
  }
  if (typeof options.extensions === "string") {
    options.extensions = [options.extensions];
  }
  options.globalName = utils.isNonEmptyString(options.globalName) && /^[a-zA-Z]+$/.test(options.globalName) ? options.globalName.toLowerCase() : `nuxt`;
  options.rootDir = utils.isNonEmptyString(options.rootDir) ? path.resolve(options.rootDir) : process.cwd();
  const nuxtConfig = getDefaultNuxtConfig();
  nuxtConfig.build._publicPath = nuxtConfig.build.publicPath;
  if (options.build && !options.build.publicPath) {
    options.build.publicPath = void 0;
  }
  lodash.defaultsDeep(options, nuxtConfig);
  options.target = options.target || "server";
  if (!Object.values(utils.TARGETS).includes(options.target)) {
    consola.warn(`Unknown target: ${options.target}. Falling back to server`);
    options.target = "server";
  }
  if (options.mode) {
    if (options.mode === utils.MODES.universal && options.ssr || options.mode === utils.MODES.spa && !options.ssr) {
      consola.warn("`mode` option is deprecated. You can safely remove it from `nuxt.config`");
    } else {
      consola.warn("`mode` option is deprecated. Please use `ssr: true` for universal mode or `ssr: false` for spa mode and remove `mode` from `nuxt.config`");
    }
  } else {
    options.mode = utils.MODES.universal;
  }
  if (options.ssr === false) {
    options.mode = utils.MODES.spa;
  }
  const modePreset = options.modes[options.mode || utils.MODES.universal];
  if (!modePreset) {
    consola.warn(`Unknown mode: ${options.mode}. Falling back to ${utils.MODES.universal}`);
  }
  lodash.defaultsDeep(options, modePreset || options.modes[utils.MODES.universal]);
  options.router.base = ufo.withTrailingSlash(ufo.normalizeURL(options.router.base));
  if (options.export) {
    consola.warn("export option is deprecated and will be removed in a future version! Please switch to generate");
    options.generate = defu.defu(options.export, options.generate);
  }
  exports.export = options.generate;
  const hasSrcDir = utils.isNonEmptyString(options.srcDir);
  const hasGenerateDir = utils.isNonEmptyString(options.generate.dir);
  options.srcDir = hasSrcDir ? path.resolve(options.rootDir, options.srcDir) : options.rootDir;
  options.buildDir = path.resolve(options.rootDir, options.buildDir);
  const { rootDir, srcDir, dir: { assets: assetsDir, static: staticDir } } = options;
  options.alias = {
    "~~": rootDir,
    "@@": rootDir,
    "~": srcDir,
    "@": srcDir,
    [assetsDir]: path.join(srcDir, assetsDir),
    [staticDir]: path.join(srcDir, staticDir),
    ...options.alias
  };
  if (!options._nuxtConfigFile) {
    options._nuxtConfigFile = path.resolve(options.rootDir, `${defaultNuxtConfigFile}.js`);
  }
  if (!options._nuxtConfigFiles) {
    options._nuxtConfigFiles = [
      options._nuxtConfigFile
    ];
  }
  options.watch.push(...options._nuxtConfigFiles);
  utils.guardDir(options, "rootDir", "buildDir");
  if (hasGenerateDir) {
    options.generate.dir = path.resolve(options.rootDir, options.generate.dir);
    utils.guardDir(options, "rootDir", "generate.dir");
  }
  if (hasSrcDir) {
    utils.guardDir(options, "srcDir", "buildDir");
    if (hasGenerateDir) {
      utils.guardDir(options, "srcDir", "generate.dir");
    }
  }
  options.modulesDir = lodash.uniq(
    utils.getMainModule().paths.concat(
      [].concat(options.modulesDir).map((dir) => path.resolve(options.rootDir, dir))
    )
  );
  const mandatoryExtensions = ["js", "mjs"];
  options.extensions = mandatoryExtensions.filter((ext) => !options.extensions.includes(ext)).concat(options.extensions);
  if (options.appTemplatePath === void 0) {
    options.appTemplatePath = path.resolve(options.buildDir, "views/app.template.html");
    if (fs.existsSync(path.join(options.srcDir, "app.html"))) {
      options.appTemplatePath = path.join(options.srcDir, "app.html");
    }
  } else {
    options.appTemplatePath = path.resolve(options.srcDir, options.appTemplatePath);
  }
  options.build.publicPath = options.build.publicPath.replace(/([^/])$/, "$1/");
  options.build._publicPath = options.build._publicPath.replace(/([^/])$/, "$1/");
  if (options.dev && utils.isUrl(options.build.publicPath)) {
    options.build.publicPath = options.build._publicPath;
  }
  if (options.store !== false && fs.existsSync(path.join(options.srcDir, options.dir.store)) && fs.readdirSync(path.join(options.srcDir, options.dir.store)).find((filename) => filename !== "README.md" && filename[0] !== ".")) {
    options.store = true;
  }
  if (options.loadingIndicator) {
    if (!utils.isPureObject(options.loadingIndicator)) {
      options.loadingIndicator = { name: options.loadingIndicator };
    }
    options.loadingIndicator = Object.assign(
      {
        name: "default",
        color: options.loading && options.loading.color || "#D3D3D3",
        color2: "#F5F5F5",
        background: options.manifest && options.manifest.theme_color || "white",
        dev: options.dev,
        loading: options.messages.loading
      },
      options.loadingIndicator
    );
  }
  if (options.debug === void 0) {
    options.debug = options.dev;
  }
  if (options.render.etag) {
    const { hash } = options.render.etag;
    if (hash) {
      const isFn = typeof hash === "function";
      if (!isFn) {
        options.render.etag.hash = void 0;
        if (options.dev) {
          consola.warn(`render.etag.hash should be a function, received ${typeof hash} instead`);
        }
      }
    }
  }
  if (options.render.csp) {
    options.render.csp = defu.defu(options.render.csp, {
      hashAlgorithm: "sha256",
      allowedSources: void 0,
      policies: void 0,
      addMeta: Boolean(options.target === utils.TARGETS.static),
      unsafeInlineCompatibility: false,
      reportOnly: options.debug
    });
    if (options.render.csp.unsafeInlineCompatiblity) {
      consola.warn("Using `unsafeInlineCompatiblity` is deprecated and will be removed in Nuxt 3. Use `unsafeInlineCompatibility` instead.");
      options.render.csp.unsafeInlineCompatibility = options.render.csp.unsafeInlineCompatiblity;
      delete options.render.csp.unsafeInlineCompatiblity;
    }
  }
  if (options.build.cssSourceMap === void 0) {
    options.build.cssSourceMap = options.dev;
  }
  const babelConfig = options.build.babel;
  if (babelConfig.cacheDirectory === void 0) {
    babelConfig.cacheDirectory = options.dev;
  }
  if (Array.isArray(babelConfig.presets)) {
    const warnPreset = (presetName) => {
      const oldPreset = "@nuxtjs/babel-preset-app";
      const newPreset = "@nuxt/babel-preset-app";
      if (presetName.includes(oldPreset)) {
        presetName = presetName.replace(oldPreset, newPreset);
        consola.warn("@nuxtjs/babel-preset-app has been deprecated, please use @nuxt/babel-preset-app.");
      }
      return presetName;
    };
    babelConfig.presets = babelConfig.presets.map((preset) => {
      const hasOptions = Array.isArray(preset);
      if (hasOptions) {
        preset[0] = warnPreset(preset[0]);
      } else if (typeof preset === "string") {
        preset = warnPreset(preset);
      }
      return preset;
    });
  }
  const vueConfig = options.vue.config;
  if (vueConfig.silent === void 0) {
    vueConfig.silent = !options.dev;
  }
  if (vueConfig.performance === void 0) {
    vueConfig.performance = options.dev;
  }
  const eligibleEnvVariables = lodash.pick(process.env, Object.keys(process.env).filter((k) => k.startsWith("NUXT_ENV_")));
  Object.assign(options.env, eligibleEnvVariables);
  options.ignore = options.ignore ? [].concat(options.ignore) : [];
  if (typeof options.ignorePrefix === "string") {
    options.ignore.push(`**/${options.ignorePrefix}*.*`);
  }
  if (options.render.gzip) {
    consola.warn("render.gzip is deprecated and will be removed in a future version! Please switch to render.compressor");
    options.render.compressor = options.render.gzip;
    delete options.render.gzip;
  }
  if (options.render.ssr === false && options.pageTransition) {
    options.pageTransition.appear = true;
  }
  options.render.ssrLog = options.dev ? options.render.ssrLog === void 0 || options.render.ssrLog : false;
  if (options.generate.fallback === true) {
    options.generate.fallback = "404.html";
  }
  if (options.build.stats === "none" || options.build.quiet === true) {
    options.build.stats = false;
  }
  if (typeof options.build.vendor !== "undefined") {
    delete options.build.vendor;
    consola.warn("vendor has been deprecated due to webpack4 optimization");
  }
  if (options.build.extractCSS && options.build.parallel) {
    options.build.parallel = false;
    consola.warn("extractCSS cannot work with parallel build due to limited work pool in thread-loader");
  }
  if (typeof options.build.extractCSS.allChunks !== "undefined") {
    consola.warn("build.extractCSS.allChunks has no effect from v2.0.0. Please use build.optimization.splitChunks settings instead.");
  }
  if (typeof options.devModules !== "undefined") {
    consola.warn("`devModules` has been renamed to `buildModules` and will be removed in Nuxt 3.");
    options.buildModules.push(...options.devModules);
    delete options.devModules;
  }
  if (options.build.optimization.minimize === void 0) {
    options.build.optimization.minimize = !options.dev;
  }
  if (options.build.optimizeCSS === void 0) {
    options.build.optimizeCSS = options.build.extractCSS ? {} : false;
  }
  const { loaders } = options.build;
  const vueLoader = loaders.vue;
  if (vueLoader.productionMode === void 0) {
    vueLoader.productionMode = !options.dev;
  }
  const styleLoaders = [
    "css",
    "cssModules",
    "less",
    "sass",
    "scss",
    "stylus",
    "vueStyle"
  ];
  for (const name of styleLoaders) {
    const loader = loaders[name];
    if (loader && loader.sourceMap === void 0) {
      loader.sourceMap = Boolean(options.build.cssSourceMap);
    }
  }
  options.build.transpile = [].concat(options.build.transpile || []);
  if (options.build.quiet === true) {
    consola.level = 0;
  }
  const { bundleRenderer } = options.render;
  if (typeof bundleRenderer.runInNewContext === "undefined") {
    bundleRenderer.runInNewContext = options.dev;
  }
  if (options.build.crossorigin) {
    consola.warn("Using `build.crossorigin` is deprecated and will be removed in Nuxt 3. Please use `render.crossorigin` instead.");
    options.render.crossorigin = options.build.crossorigin;
    delete options.build.crossorigin;
  }
  if ((_a = options.build.postcss) == null ? void 0 : _a.plugins) {
    consola.warn("`postcss.plugins` option has been moved to `postcss.postcssOptions.plugins` for aligning `postcss-loader` format.");
  }
  if (options.buildModules && options.buildModules.includes("@nuxt/postcss8")) {
    consola.info("`@nuxt/postcss8` is disabled since nuxt has upgraded to postcss v8.");
    options.buildModules = options.buildModules.filter((module) => module !== "@nuxt/postcss8");
  }
  const { timing } = options.server;
  if (timing) {
    options.server.timing = { total: true, ...timing };
  }
  if (utils.isPureObject(options.serverMiddleware)) {
    options.serverMiddleware = Object.entries(options.serverMiddleware).map(([path2, handler]) => ({ path: path2, handler }));
  }
  const useCDN = utils.isUrl(options.build.publicPath) && !options.dev;
  const isRelativePublicPath = ufo.isRelative(options.build.publicPath);
  options.app = defu.defu(options.app, {
    basePath: options.router.base,
    assetsPath: isRelativePublicPath ? options.build.publicPath : useCDN ? "/" : ufo.joinURL(options.router.base, options.build.publicPath),
    cdnURL: useCDN ? options.build.publicPath : null
  });
  options.publicRuntimeConfig = options.publicRuntimeConfig || {};
  options.publicRuntimeConfig._app = options.app;
  const { staticAssets } = options.generate;
  if (!staticAssets.version) {
    staticAssets.version = String(Math.round(Date.now() / 1e3));
  }
  if (!staticAssets.base) {
    staticAssets.base = ufo.joinURL(options.app.assetsPath, staticAssets.dir);
  }
  if (!staticAssets.versionBase) {
    staticAssets.versionBase = ufo.joinURL(staticAssets.base, staticAssets.version);
  }
  const isJest = typeof jest !== "undefined";
  const defaultCreateRequire = isJest ? "native" : "jiti";
  options.createRequire = process.env.NUXT_CREATE_REQUIRE || options.createRequire || defaultCreateRequire;
  if (options.createRequire === "native" || options.createRequire === "jiti") {
    const useJiti = options.createRequire === "jiti";
    options.createRequire = (p) => utils.createRequire(typeof p === "string" ? p : p.filename, useJiti);
  } else if (typeof options.createRequire !== "function") {
    throw new TypeError(
      `Unsupported createRequire value ${options.createRequire}! Possible values: "native", "jiti", <Function>`
    );
  }
  if (options.build.indicator === true) {
    options.build.indicator = nuxtConfig.build.indicator;
  }
  if (!options.dev || !options._cli || !utils.getPKG("@nuxt/loading-screen")) {
    options.build.loadingScreen = false;
  }
  if (options.build.loadingScreen) {
    options._modules.push(["@nuxt/loading-screen", options.build.loadingScreen]);
  } else {
    options.build.indicator = false;
  }
  if (parseInt(process.versions.node.slice(0, 2)) > 16) {
    options.buildModules.push(function patchMD4() {
      const crypto = require("crypto");
      const _createHash = crypto.createHash;
      crypto.createHash = function(algorithm, options2) {
        return _createHash(algorithm === "md4" ? "md5" : algorithm, options2);
      };
    });
  }
  if (!options._start && utils.getPKG("@nuxt/components")) {
    options._modules.push("@nuxt/components");
  }
  if (options.telemetry !== false && !options.test && !destr(process.env.NUXT_TELEMETRY_DISABLED) && utils.getPKG("@nuxt/telemetry")) {
    options._modules.push("@nuxt/telemetry");
  }
  return options;
}

async function loadNuxtConfig({
  rootDir = ".",
  envConfig = {},
  configFile = defaultNuxtConfigFile,
  configContext = {},
  configOverrides = {}
} = {}) {
  var _a, _b, _c, _d, _e;
  rootDir = path.resolve(rootDir);
  const _require = utils.createRequire(rootDir, true);
  let options = {};
  try {
    configFile = _require.resolve(path.resolve(rootDir, configFile));
  } catch (e) {
    if (e.code !== "MODULE_NOT_FOUND") {
      throw e;
    } else if (configFile !== defaultNuxtConfigFile) {
      consola.fatal("Config file not found: " + configFile);
    }
    configFile = void 0;
  }
  envConfig = {
    dotenv: ".env",
    env: process.env,
    expand: true,
    ...envConfig
  };
  const env = loadEnv(envConfig, rootDir);
  for (const key in env) {
    if (!key.startsWith("_") && envConfig.env[key] === void 0) {
      envConfig.env[key] = env[key];
    }
  }
  if (configFile) {
    utils.clearRequireCache(configFile);
    options = _require(configFile) || {};
    if (options.default) {
      options = options.default;
    }
    if (typeof options === "function") {
      try {
        options = await options(configContext);
        if (options.default) {
          options = options.default;
        }
      } catch (error) {
        consola.error(error);
        consola.fatal("Error while fetching async configuration");
      }
    }
    options = { ...options };
    options._nuxtConfigFile = configFile;
    options._nuxtConfigFiles = Array.from(utils.scanRequireTree(configFile));
    if (!options._nuxtConfigFiles.includes(configFile)) {
      options._nuxtConfigFiles.unshift(configFile);
    }
  }
  if (typeof options.rootDir !== "string") {
    options.rootDir = rootDir;
  }
  const dev = (_b = (_a = configOverrides.dev) != null ? _a : options.dev) != null ? _b : configContext.dev;
  const buildDir = (_e = (_d = (_c = configOverrides.buildDir) != null ? _c : options.buildDir) != null ? _d : configContext.buildDir) != null ? _e : ".nuxt";
  options = defu.defu(
    configOverrides,
    options,
    !dev ? rc__namespace.read({ name: "nuxtrc", dir: path.resolve(buildDir, "dist") }) : {},
    rc__namespace.read({ name: ".nuxtrc", dir: options.rootDir }),
    rc__namespace.readUser(".nuxtrc")
  );
  options._env = env;
  options._envConfig = envConfig;
  if (configContext) {
    configContext.env = env;
  }
  if (envConfig.expand) {
    for (const c of ["publicRuntimeConfig", "privateRuntimeConfig"]) {
      if (options[c]) {
        if (typeof options[c] === "function") {
          options[c] = options[c](env);
        }
        expand(options[c], env, destr);
      }
    }
  }
  return options;
}
function loadEnv(envConfig, rootDir = process.cwd()) {
  const env = /* @__PURE__ */ Object.create(null);
  if (envConfig.dotenv) {
    envConfig.dotenv = path.resolve(rootDir, envConfig.dotenv);
    if (fs.existsSync(envConfig.dotenv)) {
      const parsed = dotenv.parse(fs.readFileSync(envConfig.dotenv, "utf-8"));
      Object.assign(env, parsed);
    }
  }
  if (!envConfig.env._applied) {
    Object.assign(env, envConfig.env);
    envConfig.env._applied = true;
  }
  if (envConfig.expand) {
    expand(env);
  }
  return env;
}
function expand(target, source = {}, parse = (v) => v) {
  function getValue(key) {
    return source[key] !== void 0 ? source[key] : target[key];
  }
  function interpolate(value, parents = []) {
    if (typeof value !== "string") {
      return value;
    }
    const matches = value.match(/(.?\${?(?:[a-zA-Z0-9_:]+)?}?)/g) || [];
    return parse(matches.reduce((newValue, match) => {
      const parts = /(.?)\${?([a-zA-Z0-9_:]+)?}?/g.exec(match);
      const prefix = parts[1];
      let value2, replacePart;
      if (prefix === "\\") {
        replacePart = parts[0];
        value2 = replacePart.replace("\\$", "$");
      } else {
        const key = parts[2];
        replacePart = parts[0].substring(prefix.length);
        if (parents.includes(key)) {
          consola.warn(`Please avoid recursive environment variables ( loop: ${parents.join(" > ")} > ${key} )`);
          return "";
        }
        value2 = getValue(key);
        value2 = interpolate(value2, [...parents, key]);
      }
      return value2 !== void 0 ? newValue.replace(replacePart, value2) : newValue;
    }, value));
  }
  for (const key in target) {
    target[key] = interpolate(getValue(key));
  }
}

exports.defaultNuxtConfigFile = defaultNuxtConfigFile;
exports.getDefaultNuxtConfig = getDefaultNuxtConfig;
exports.getNuxtConfig = getNuxtConfig;
exports.loadNuxtConfig = loadNuxtConfig;
